using System;
using System.IO;
using System.Xml;
using System.Collections;

using Bertaccini.Utils;

using Team_Project;
using Team_Project.Elements;
using Team_Project.PersistencyManagers.Protocols;

namespace Team_Project.FileModule
{
	public class FileModule : ITPModule, IOperationHandler
	{
		protected static readonly string OldDir = "Olds.TP";

		protected object chTkn;
		public FileModule()
		{}

		#region IOperationHandler Members
		public object ChainToken
		{
			get{return chTkn;}
			set{chTkn = value;}
		}

		public string Compute(IDictionary data,string fullXml, out bool ok)
		{
			throw new Team_Project.Exceptions.TeamProjectException("File Module cannot be called directly");
		}

		public System.IO.Stream Compute(IDictionary data, XmlTextReader tr,out bool ok)
		{
			ok = false;
			string userID, role, project, operation;
			userID = (string)data["userID"];
			role = (string)data["role"];
			project = (string)data["project"];
			operation = (string)data["operation"];
			Stream res = null;
			//XmlTextReader tr = new XmlTextReader(data);
			if(!tr.Read() || tr.Name != "ActionParameters")
				return TPEngine.FormatError("ActionParameters");
			bool err = true;
			switch(operation)
			{
				case "FileStore":
					res = FileStore(userID, role, project, operation, tr,out err);
					break;
				case "FileRetrive":
					res = FileRetrive(userID, role, project, operation,tr,out err);
					break;
				case "FileDelete":
					res = FileDelete(userID, role, project, operation,tr,out err);
					break;
				case "GetFileInfo":
					  res = GetFileInfo(userID, role, project, operation,tr,out err);
					  break;
				case "CreateLocation":
					res = CreateLocation(userID, role, project, operation,tr,out err);
					break;
				case "ListElements":
					res = ListFiles(userID, role, project, operation,tr,out err);
					break;
				case "ListLocations":
					res = ListLocations(userID, role, project, operation,tr,out err);
					break;
				case "LockFile":
					res = LockFile(userID, role, project, operation,tr,out err);
					break;
				case "UnlockFile":
					res = UnLockFile(userID, role, project, operation,tr,out err);
					break;
			}
			if(err) return res;
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("ActionParameters end element");
			ok = true;
			return res;
		}

		#endregion
		#region Common things
		protected Stream ReadLocation(XmlTextReader tr, out string loc)
		{
			loc = null;
			if(!tr.Read() || tr.Name != "LocationInfo")
				return TPEngine.FormatError("LocationInfo");
			if(!tr.Read() || tr.Name != "LocationName")
				return TPEngine.FormatError("LocationName");
			if(!tr.Read() || tr.NodeType != XmlNodeType.Text)
				return TPEngine.FormatError("LocationName text");
			loc = tr.Value;
			if(loc == string.Empty)
				return TPEngine.FormatError("LocationName not empty");
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("LocationName end element");
			return null;
		}
		protected Stream ReadData(XmlTextReader tr, out string fName,out string fPath)
		{
			fName = null;
			fPath = null;
			if(!tr.Read() || tr.Name != "FileInfo")
				return TPEngine.FormatError("FileInfo");
			if(!tr.Read() || tr.Name != "FileName")
				return TPEngine.FormatError("FileName");
			if(!tr.Read() || tr.NodeType != XmlNodeType.Text)
				return TPEngine.FormatError("FileName text");
			fName = tr.Value;
			if(fName == string.Empty)
				return TPEngine.FormatError("FileName not empty");
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("FileName end element");
			if(!tr.Read() || tr.Name != "FilePath")
				return TPEngine.FormatError("FilePath");
			if(!tr.Read())
				return TPEngine.FormatError("FilePath text or end element");
			fPath = "";
			if(tr.NodeType == XmlNodeType.Text)
			{
				fPath = tr.Value;
				if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
					return TPEngine.FormatError("FilePath end element");
			}
			return null;
		}
		protected Stream ReadDataHash(XmlTextReader tr, out string fName,out string fPath,out byte[] hsh,bool hashMandatory)
		{
			hsh = null;
			Stream s = ReadData(tr,out fName,out fPath);
			if(s != null) return s;
			if(!tr.Read() || tr.Name != "FileHash")
			{
				if(hashMandatory)
					return TPEngine.FormatError("FileHash");
				else return null;
				
			}
			if(!tr.Read() || tr.NodeType != XmlNodeType.Text)
				return TPEngine.FormatError("FileHash text");
			string hashStng = tr.Value;
			hsh = null;
			try
			{
				hsh = Convert.FromBase64String(hashStng);
				if(hsh.Length != 20) throw new Exception("Wrong hash length: is " + hsh.Length + " should be 20");
			}
			catch(Exception e)
			{return TPEngine.ExceptionError("reading hash in request",e);}
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("FileHash end element");
			return null;
		}

		protected LockElement GetLock(string project,string path,string name,out bool fileExists)
		{
			fileExists = true;
			IPersistencyProtocol ppL = Globals.Instance.Protocols.GetProtocolFor(project,typeof(LockElement),Direction.Retrive);
			IPersistencyProtocol ppF = Globals.Instance.Protocols.GetProtocolFor(project,typeof(FileElement),Direction.Store);
			ppF.Intit();
			ppL.Intit();
			
			if(ppF.QueryElementExists(path,name))
			{
				if(ppL.QueryElementExists(path,name))
					return (LockElement)ppL.Retrive(path,name,typeof(LockElement));
			}
			else fileExists = false;
			return null;

		}
		protected Stream CheckLock(string project,string path,string name,string userID)
		{
			LockElement lk;
			try
			{
				lock(Globals.Instance.Protocols)
				{
					bool exist;
					lk = GetLock(project,path,name,out exist);
					if(exist)
					{
						if(lk != null && lk.LockerID != userID)
							return LockError();
					}
				}
			}
			catch(Exception LockE)
			{
				return TPEngine.ExceptionError("checking ownership of the lock",LockE);
			}
			return null;
		}

		protected void WirteFileInfo(FileElement f, XmlTextWriter tw)
		{
			tw.WriteStartElement("File");
			tw.WriteStartElement("Hash");
			tw.WriteString(Convert.ToBase64String(f.Hash,0,f.Hash.Length));
			tw.WriteEndElement();
			tw.WriteStartElement("Uploader");
			tw.WriteString(f.UploaderID);
			tw.WriteEndElement();
			tw.WriteStartElement("UploadTime");
			tw.WriteString(f.UploadTime.ToString());
			tw.WriteEndElement();
			tw.WriteEndElement();
		}
		#endregion

		public Stream FileStore(string userID, string role, string project, string operation, XmlTextReader tr,out bool err)
		{
			err = true;
			#region Read Data
			string name, path;
			Stream strRes = ReadData(tr,out name,out path);
			if(strRes != null) return strRes;
			#endregion
			//TODO: check authorization
			IPersistencyProtocol ppF  = Globals.Instance.Protocols.GetProtocolFor(project,typeof(FileElement),Direction.Store);
			ppF.Intit();
			#region Check if user owns the lock
			strRes = CheckLock(project,path,name,userID);
			if(strRes != null) return strRes;
			#endregion
			#region Retrive raw data
			if(!tr.Read() || tr.Name != "FileData")
				return TPEngine.FormatError("FileData");

			if(!tr.Read() || tr.NodeType != XmlNodeType.Text)
				return TPEngine.FormatError("File content");
			string s = tr.Value.Trim();
			byte[] fc = Convert.FromBase64String(s);
			MemoryStream ls = new MemoryStream(fc);
			#endregion

			#region Store of the file
			FileElement fe = null;
			if(ppF.QueryElementExists(path,name))
			{
				FileElement old = null;
				try
				{
					IPersistencyProtocol ppFR = Globals.Instance.Protocols.GetProtocolFor(project,typeof(FileElement),Direction.Retrive);
					ppFR.Intit();
					fe = (FileElement)ppFR.Retrive(path,name,typeof(FileElement));
					old = new FileElement(project,path + "\\" + OldDir,name);
					fe.BindForCopy(old);
					if(!ppF.QueryLocationExists(old.Location))
						ppF.CreateLocation(old.Location);
					ppF.Store(old);
				}
				catch(Exception)	{}
				finally{
					if(fe != null)
					{
						fe.Dispose();
						fe = null;
					}
					if(old != null) old.Dispose();
				}
			}

			try
			{
				fe = new FileElement(project,path,name);
				fe.UploaderID = userID;
				fe.UploadTime = DateTime.Now;
				fe.SetRawData(ls);
				ppF.Store(fe);
			}
			catch(Exception fileE)
			{
				return TPEngine.ExceptionError("storing file",fileE);
			}
			finally
			{
				if(fe != null) fe.Dispose();
			}
			#endregion
			#region Consume of end elements
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("FileData end element");
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("FileInfo end element");
			#endregion
			
			#region Wirte Response
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("FileStroreResponse");
			tw.WriteStartElement("Hash");
			tw.WriteString(Convert.ToBase64String(fe.Hash,0,fe.Hash.Length));
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			err = false;
			return ms;
			#endregion

		}

		public Stream FileRetrive(string userID, string role, string project, string operation, XmlTextReader tr,out bool err)
		{
			err = true;
			#region Read Data
			string name, path;
			byte[] Hash;
			Stream sres = ReadDataHash(tr,out name,out path,out Hash,true);
			if(sres != null) return sres;
			#endregion
			//TODO: check authorization
			#region Retrive of the file
			IPersistencyProtocol ppF = Globals.Instance.Protocols.GetProtocolFor(project,typeof(FileElement),Direction.Retrive);
			ppF.Intit();
			FileElement fe = null;
			try
			{
				if(ppF.QueryElementExists(path,name))
				{
					fe = (FileElement)ppF.Retrive(path,name,typeof(FileElement));
					if(!CompareHash(fe.Hash,Hash))
					{
						fe = (FileElement)ppF.Retrive(path+"\\"+OldDir,name,typeof(FileElement));
						if(!CompareHash(fe.Hash,Hash))
							throw new Exception("File with specified hash not exists");
					}
				}
			}
			catch(Exception fileE)
			{
				if(fe != null) fe.Dispose();
				return TPEngine.ExceptionError("retriving file",fileE);
			}
			#endregion
			#region Consume of end elements
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("FileInfo end element");
			#endregion
			
			#region Wirte Response
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("FileRetriveResponse");
			tw.WriteStartElement("Data");
			MemoryStream msD = new MemoryStream();
			fe.WriteTo(msD);
			tw.WriteString(Convert.ToBase64String(msD.GetBuffer(),0,(int)msD.Length));
			msD.Close(); fe.Dispose();
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			err = false;
			return ms;
			#endregion
		}

		public Stream FileDelete(string userID, string role, string project, string operation, XmlTextReader tr,out bool err)
		{
			err = true;
			#region Read Data
			string name, path;
			byte[] Hash;
			Stream sres = ReadDataHash(tr,out name,out path,out Hash,true);
			if(sres != null) return sres;
			#endregion
			//TODO: check authorization
			#region Check if the user owns the lock
			sres = CheckLock(project,path,name,userID);
			if(sres != null) return sres;
			#endregion
			#region Delete of the file
			IPersistencyProtocol ppF = Globals.Instance.Protocols.GetProtocolFor(project,typeof(FileElement),Direction.Store);
			ppF.Intit();
			FileElement fe = null;
			try
			{
				if(ppF.QueryElementExists(path,name))
				{
					fe = (FileElement)ppF.Retrive(path,name,typeof(FileElement));
					if(CompareHash(fe.Hash,Hash))
					{
						ppF.DestroyElement(fe);
						fe.Dispose();
						if(ppF.QueryElementExists(path+"\\"+OldDir,name))
						{
							fe = (FileElement)ppF.Retrive(path+"\\"+OldDir,name,typeof(FileElement));
							ppF.DestroyElement(fe);
							fe.Dispose();
						}
					}
					else
					{
						if(ppF.QueryElementExists(path+"\\"+OldDir,name))
						{
							fe = (FileElement)ppF.Retrive(path+"\\"+OldDir,name,typeof(FileElement));
							if(CompareHash(fe.Hash,Hash))
							{
								ppF.DestroyElement(fe);
								fe.Dispose();
							}
							else
								throw new Exception("File with specified hash not exists");
						}
					}
				}
			}
			catch(Exception fileE)
			{
				if(fe != null) fe.Dispose();
				return TPEngine.ExceptionError("deleting file",fileE);
			}
			try
			{
				IPersistencyProtocol ppL = Globals.Instance.Protocols.GetProtocolFor(project,typeof(LockElement),Direction.Store);
				LockElement le = (LockElement)ppL.Retrive(path,name,typeof(LockElement));
				ppL.DestroyElement(le);
			}
			catch(Exception){}
			#endregion
			#region Consume of end elements
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("FileInfo end element");
			#endregion
			
			#region Wirte Response
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("FileDeleteResponse");
			tw.WriteString("OK");
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			err = false;
			return ms;
			#endregion
		}

		public Stream GetFileInfo(string userID, string role, string project, string operation, XmlTextReader tr,out bool err)
		{
			err = false;
			#region ReadData
			string name, path;
			string status;
			byte[] hsh;
			Stream strRes = ReadDataHash(tr,out name,out path,out hsh,false);
			if(strRes != null) return strRes;
			if (hsh != null)
				status = "Unknown";
			else status = "Missing";
			#endregion
			#region Retrive of the file
			FileElement fe = null;
			LockElement le = null;
			FileElement[] history = null;
			IPersistencyProtocol ppF = null;
			
			try
			{
				ppF = Globals.Instance.Protocols.GetProtocolFor(project,typeof(FileElement),Direction.Retrive);
				ppF.Intit();
				if(ppF.QueryElementExists(path,name))
				{
					fe = (FileElement) ppF.Retrive(path,name,typeof(FileElement));
					if(hsh != null && CompareHash(hsh,fe.Hash))
						status = "Current";
					if(ppF.QueryElementExists(path+"\\"+OldDir,name))
					{
						history = new FileElement[1];
						history[0] = (FileElement)ppF.Retrive(path+"\\"+OldDir,name,typeof(FileElement));
						if(hsh != null && status != "Current" && CompareHash(hsh,history[0].Hash))
							status = "OutOfDate";
					}
					IPersistencyProtocol ppL = Globals.Instance.Protocols.GetProtocolFor(project,typeof(LockElement),Direction.Retrive);
					ppL.Intit();
					if(ppL.QueryElementExists(path,name))
					{
						le = (LockElement)ppL.Retrive(path,name,typeof(LockElement));
					}
				}
			}
			catch(Exception e)
			{
				if(fe != null) fe.Dispose();
				if(history != null)
					foreach(FileElement ff in history) ff.Dispose();
				return TPEngine.ExceptionError("retriving file",e);
			}
			#endregion
			#region Read End elements
			if(tr.Name == "FileHash" && tr.NodeType == XmlNodeType.EndElement)
				if(!tr.Read())
					return TPEngine.FormatError("FileInfo end element");
			if(tr.Name != "FileInfo" || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("FileInfo end element");
			#endregion
			#region Write Response
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("GetFileInfoResponse");
			if(fe != null)
			{
				WirteFileInfo(fe,tw);
				if(history != null)
				{
					tw.WriteStartElement("History");
					foreach(FileElement fe2 in history)
						WirteFileInfo(fe2,tw);
					tw.WriteEndElement();
				}
				if(hsh != null)
				{
					tw.WriteStartElement("Status");
					tw.WriteString(status);
					tw.WriteEndElement();
				}
				if(le != null)
				{
					tw.WriteStartElement("Lock");
					tw.WriteStartElement("Owner");
					tw.WriteString(le.LockerID);
					tw.WriteEndElement();
					tw.WriteStartElement("Time");
					tw.WriteString(le.LockTime.ToString());
					tw.WriteEndElement();
					tw.WriteStartElement("RequestID");
					tw.WriteString(le.RequestID);
					tw.WriteEndElement();
					tw.WriteEndElement();
				}
			}
			else
			{
				tw.WriteString("File not exists");
			}
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			err = false;
			return ms;
			#endregion
		}
		
		public Stream CreateLocation(string userID, string role, string project,string operation, XmlTextReader tr,out bool err)
		{
			err = true;
			string LocName;
			#region Read Data
			Stream strRes = ReadLocation(tr,out LocName);
			if(strRes != null) return strRes;
			#endregion
            
			IPersistencyProtocol pp = Globals.Instance.Protocols.GetProtocolFor(project,typeof(FileElement),Direction.Store);
			pp.Intit();
			if(!pp.QueryLocationExists(LocName))
				pp.CreateLocation(LocName);
			#region Wirte Response
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("CreateLocationResponse");
			tw.WriteString("OK");
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			err = false;
			return ms;
			#endregion
		}
		public Stream ListFiles(string userID, string role, string project,string operation, XmlTextReader tr,out bool err)
		{
			err = true;
			string LocName;
			#region Read Data
			Stream strRes = ReadLocation(tr,out LocName);
			if(strRes != null) return strRes;
			#endregion
            
			IPersistencyProtocol pp = Globals.Instance.Protocols.GetProtocolFor(project,typeof(FileElement),Direction.Store);
			pp.Intit();
			if(!pp.QueryLocationExists(LocName))
				return TPEngine.CustomError("Location not exists");
			#region Wirte Response
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("ListElementsResponse");
			foreach(string sne in pp.QueryElemensInLocation(LocName))
			{
				tw.WriteStartElement("Element");
				tw.WriteString(sne);
				tw.WriteEndElement();
			}
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			err = false;
			return ms;
			#endregion
		}
		
		public Stream LockFile(string userID, string role, string project, string operation, XmlTextReader tr,out bool err)
		{
			err = true;
			#region Read Data
			string name, path;
			byte[] Hash;
			Stream sres = ReadDataHash(tr,out name,out path,out Hash,true);
			if(sres != null) return sres;
			#endregion
			//TODO: check authorization
			#region Check there's a lock on the file
			LockElement lk = null;
			bool Fexist;
			try
			{
				lk = GetLock(project,path,name,out Fexist);
				if(!Fexist)
					return TPEngine.CustomError("File to lock does not exists");
				if(lk != null)
				{
					if (lk.LockerID != userID)
						return TPEngine.CustomError("File locked by another user");
				}
				else
				{
					//TODO check if file has the right hash
					try
					{
						lk = new LockElement(project,path,name);
						lk.LockerID = userID;
						lk.LockTime = DateTime.Now;
						IPersistencyProtocol lpp = Globals.Instance.Protocols.GetProtocolFor(
							project,typeof(LockElement),Direction.Store);
						lpp.Intit();
						lpp.Store(lk);
					}
					catch(Exception lecc)
					{
						return TPEngine.ExceptionError("locking file",lecc);
					}
				}
			}
			catch(Exception ecc)
			{
				return TPEngine.ExceptionError("checking lock",ecc);
			}
			#endregion
			#region Consume of end elements
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("FileInfo end element");
			#endregion
			
			#region Wirte Response
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("LockFileResponse");
			tw.WriteString("OK");
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			err = false;
			return ms;
			#endregion
		}

		public Stream UnLockFile(string userID, string role, string project, string operation, XmlTextReader tr,out bool err)
		{
			err = true;
			#region Read Data
			string name, path;
			Stream sres = ReadData(tr,out name,out path);
			if(sres != null) return sres;
			#endregion
			//TODO: check authorization
			#region Delete the lock
			LockElement lk = null;
			bool Fexist;
			try
			{
				lk = GetLock(project,path,name,out Fexist);
				if(!Fexist)
					return TPEngine.CustomError("File to unlock does not exists");
				if(lk != null)
				{
					if (lk.LockerID != userID)
						return TPEngine.CustomError("The lock is owned by another user");
					try
					{
						IPersistencyProtocol lpp = Globals.Instance.Protocols.GetProtocolFor(
							project,typeof(LockElement),Direction.Store);
						lpp.Intit();
						lpp.DestroyElement(lk);
					}
					catch(Exception lecc)
					{
						return TPEngine.ExceptionError("unlocking file",lecc);
					}
				}
			}
			catch(Exception ecc)
			{
				return TPEngine.ExceptionError("checking lock",ecc);
			}
			#endregion
			#region Consume of end elements
			if(!tr.Read() || tr.NodeType != XmlNodeType.EndElement)
				return TPEngine.FormatError("FileInfo end element");
			#endregion
			
			#region Wirte Response
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("UnlockFileResponse");
			tw.WriteString("OK");
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			err = false;
			return ms;
			#endregion
		}

		public Stream ListLocations(string userID, string role, string project,string operation, XmlTextReader tr,out bool err)
		{
			err = true;
			string LocName;
			#region Read Data
			Stream strRes = ReadLocation(tr,out LocName);
			if(strRes != null) return strRes;
			#endregion
            
			IPersistencyProtocol pp = Globals.Instance.Protocols.GetProtocolFor(project,typeof(FileElement),Direction.Retrive);
			pp.Intit();
			if(LocName != "\\" && !pp.QueryLocationExists(LocName))
				return TPEngine.CustomError("Location not exists");
			if(LocName == "\\") LocName = "";
			#region Wirte Response
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("ListLocationssResponse");
			foreach(string sne in pp.QueryLocations(LocName))
			{
				if(sne == OldDir) continue;
				tw.WriteStartElement("Location");
				tw.WriteString(sne);
				tw.WriteEndElement();
			}
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			err = false;
			return ms;
			#endregion
		}
		
		
		public static bool CompareHash(byte[] a1, byte[] a2)
		{
			for(int i = 0; i< a1.Length;i++)
			{
				if(a1[i] != a2[i]) return false;
			}
			return true;
		}

		#region Response
		
		protected Stream LockError()
		{
			MemoryStream ms = new MemoryStream();
			XmlTextWriter tw = new XmlTextWriter(ms,System.Text.Encoding.UTF8);
			tw.WriteStartDocument(true);
			tw.WriteStartElement("TeamProjectResponse");
			tw.WriteStartElement("Error");
			tw.WriteString("Cannot upload file: user don't own lock");
			tw.WriteEndElement();
			tw.WriteEndElement();
			tw.Flush();
			ms.Position = 0;
			return ms;
		}

		#endregion

		#region ITPModule Members

		public void RegisterOperations(TPEngine eng)
		{
			eng.Register("FileStore",this);
			eng.Register("FileRetrive",this);
			eng.Register("FileDelete",this);
			eng.Register("GetFileInfo",this);
			eng.Register("CreateLocation",this);
			eng.Register("ListElements",this);
			eng.Register("LockFile",this);
			eng.Register("ListLocations",this);
			eng.Register("UnlockFile",this);
		}

		#endregion
	}
}
